/**
 * @license Copyright (c) 2003-2023, CKSource Holding sp. z o.o. All rights reserved.
 * For licensing, see LICENSE.md or https://ckeditor.com/legal/ckeditor-oss-license
 */

/* eslint-env node */

const fetch = require( 'node-fetch' );

const {
	INTEGRATION_CI_TRAVIS_TOKEN,
	INTEGRATION_CI_ORGANIZATION,
	INTEGRATION_CI_REPOSITORY,
	TRAVIS_EVENT_TYPE,
	TRAVIS_BRANCH
} = process.env;

/**
 * This script triggers Travis that verifies whether projects that depend on CKEditor 5 build correctly.
 *
 * In order to integrate the action in a new repository, you need add a few secrets in the new repository.
 *   - INTEGRATION_CI_ORGANIZATION - a name of the organization that keeps the repository where the build should be triggered
 *   - INTEGRATION_CI_REPOSITORY - a name of the repository where the build should be triggered
 *   - INTEGRATION_CI_TRAVIS_TOKEN - an authorization token generated by Travis CLI: `travis --pro token`
 *
 * @param {String} repository A slug of repository that triggers a new build.
 * @param {String} lastCommit A hash of latest commit.
 * @returns {Object} Travis API response as JSON.
 */
module.exports = function triggerCkeditor5ContinuousIntegration( repository, lastCommit ) {
	// We want to trigger the integration build when current build was triggered by push commit or API call.
	if ( TRAVIS_EVENT_TYPE !== 'push' && TRAVIS_EVENT_TYPE !== 'api' ) {
		return;
	}

	// Trigger the integration build only when checking the "master" branch in the repository.
	if ( TRAVIS_BRANCH !== 'master' ) {
		return;
	}

	const requestUrl = `https://api.travis-ci.com/repo/${ INTEGRATION_CI_ORGANIZATION }%2F${ INTEGRATION_CI_REPOSITORY }/requests`;
	const requestOptions = {
		method: 'post',
		headers: {
			'Content-Type': 'application/json',
			'Accept': 'application/json',
			'Travis-API-Version': 3,
			'Authorization': `token ${ INTEGRATION_CI_TRAVIS_TOKEN }`
		},
		body: JSON.stringify( {
			request: {
				branch: 'master',
				message: `Repository: ${ repository }\n\nCommit: https://github.com/${ repository }/commit/${ lastCommit }.`
			}
		} )
	};

	return fetch( requestUrl, requestOptions )
		.then( res => res.json() );
};
